// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: Copyright OpenBMC Authors
#pragma once
#include <nlohmann/json.hpp>

namespace log_entry
{
// clang-format off

enum class EventSeverity{
    Invalid,
    OK,
    Warning,
    Critical,
};

enum class LogEntryType{
    Invalid,
    Event,
    SEL,
    Oem,
    CXL,
};

enum class LogEntryCode{
    Invalid,
    Assert,
    Deassert,
    LowerNoncriticalgoinglow,
    LowerNoncriticalgoinghigh,
    LowerCriticalgoinglow,
    LowerCriticalgoinghigh,
    LowerNonrecoverablegoinglow,
    LowerNonrecoverablegoinghigh,
    UpperNoncriticalgoinglow,
    UpperNoncriticalgoinghigh,
    UpperCriticalgoinglow,
    UpperCriticalgoinghigh,
    UpperNonrecoverablegoinglow,
    UpperNonrecoverablegoinghigh,
    TransitiontoIdle,
    TransitiontoActive,
    TransitiontoBusy,
    StateDeasserted,
    StateAsserted,
    PredictiveFailuredeasserted,
    PredictiveFailureasserted,
    LimitNotExceeded,
    LimitExceeded,
    PerformanceMet,
    PerformanceLags,
    TransitiontoOK,
    TransitiontoNonCriticalfromOK,
    TransitiontoCriticalfromlesssevere,
    TransitiontoNonrecoverablefromlesssevere,
    TransitiontoNonCriticalfrommoresevere,
    TransitiontoCriticalfromNonrecoverable,
    TransitiontoNonrecoverable,
    Monitor,
    Informational,
    DeviceRemovedDeviceAbsent,
    DeviceInsertedDevicePresent,
    DeviceDisabled,
    DeviceEnabled,
    TransitiontoRunning,
    TransitiontoInTest,
    TransitiontoPowerOff,
    TransitiontoOnLine,
    TransitiontoOffLine,
    TransitiontoOffDuty,
    TransitiontoDegraded,
    TransitiontoPowerSave,
    InstallError,
    FullyRedundant,
    RedundancyLost,
    RedundancyDegraded,
    NonredundantSufficientResourcesfromRedundant,
    NonredundantSufficientResourcesfromInsufficientResources,
    NonredundantInsufficientResources,
    RedundancyDegradedfromFullyRedundant,
    RedundancyDegradedfromNonredundant,
    D0PowerState,
    D1PowerState,
    D2PowerState,
    D3PowerState,
    OEM,
};

enum class SensorType{
    Invalid,
    PlatformSecurityViolationAttempt,
    Temperature,
    Voltage,
    Current,
    Fan,
    PhysicalChassisSecurity,
    Processor,
    PowerSupplyConverter,
    PowerUnit,
    CoolingDevice,
    OtherUnitsbasedSensor,
    Memory,
    DriveSlotBay,
    POSTMemoryResize,
    SystemFirmwareProgress,
    EventLoggingDisabled,
    SystemEvent,
    CriticalInterrupt,
    ButtonSwitch,
    ModuleBoard,
    MicrocontrollerCoprocessor,
    AddinCard,
    Chassis,
    ChipSet,
    OtherFRU,
    CableInterconnect,
    Terminator,
    SystemBootRestart,
    BootError,
    BaseOSBootInstallationStatus,
    OSStopShutdown,
    SlotConnector,
    SystemACPIPowerState,
    Watchdog,
    PlatformAlert,
    EntityPresence,
    MonitorASICIC,
    LAN,
    ManagementSubsystemHealth,
    Battery,
    SessionAudit,
    VersionChange,
    FRUState,
    OEM,
};

enum class LogDiagnosticDataTypes{
    Invalid,
    Manager,
    PreOS,
    OS,
    OEM,
    CPER,
    CPERSection,
    Device,
};

enum class OriginatorTypes{
    Invalid,
    Client,
    Internal,
    SupportingService,
};

enum class CXLEntryType{
    Invalid,
    DynamicCapacity,
    Informational,
    Warning,
    Failure,
    Fatal,
};

NLOHMANN_JSON_SERIALIZE_ENUM(EventSeverity, {
    {EventSeverity::Invalid, "Invalid"},
    {EventSeverity::OK, "OK"},
    {EventSeverity::Warning, "Warning"},
    {EventSeverity::Critical, "Critical"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(LogEntryType, {
    {LogEntryType::Invalid, "Invalid"},
    {LogEntryType::Event, "Event"},
    {LogEntryType::SEL, "SEL"},
    {LogEntryType::Oem, "Oem"},
    {LogEntryType::CXL, "CXL"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(LogEntryCode, {
    {LogEntryCode::Invalid, "Invalid"},
    {LogEntryCode::Assert, "Assert"},
    {LogEntryCode::Deassert, "Deassert"},
    {LogEntryCode::LowerNoncriticalgoinglow, "Lower Non-critical - going low"},
    {LogEntryCode::LowerNoncriticalgoinghigh, "Lower Non-critical - going high"},
    {LogEntryCode::LowerCriticalgoinglow, "Lower Critical - going low"},
    {LogEntryCode::LowerCriticalgoinghigh, "Lower Critical - going high"},
    {LogEntryCode::LowerNonrecoverablegoinglow, "Lower Non-recoverable - going low"},
    {LogEntryCode::LowerNonrecoverablegoinghigh, "Lower Non-recoverable - going high"},
    {LogEntryCode::UpperNoncriticalgoinglow, "Upper Non-critical - going low"},
    {LogEntryCode::UpperNoncriticalgoinghigh, "Upper Non-critical - going high"},
    {LogEntryCode::UpperCriticalgoinglow, "Upper Critical - going low"},
    {LogEntryCode::UpperCriticalgoinghigh, "Upper Critical - going high"},
    {LogEntryCode::UpperNonrecoverablegoinglow, "Upper Non-recoverable - going low"},
    {LogEntryCode::UpperNonrecoverablegoinghigh, "Upper Non-recoverable - going high"},
    {LogEntryCode::TransitiontoIdle, "Transition to Idle"},
    {LogEntryCode::TransitiontoActive, "Transition to Active"},
    {LogEntryCode::TransitiontoBusy, "Transition to Busy"},
    {LogEntryCode::StateDeasserted, "State Deasserted"},
    {LogEntryCode::StateAsserted, "State Asserted"},
    {LogEntryCode::PredictiveFailuredeasserted, "Predictive Failure deasserted"},
    {LogEntryCode::PredictiveFailureasserted, "Predictive Failure asserted"},
    {LogEntryCode::LimitNotExceeded, "Limit Not Exceeded"},
    {LogEntryCode::LimitExceeded, "Limit Exceeded"},
    {LogEntryCode::PerformanceMet, "Performance Met"},
    {LogEntryCode::PerformanceLags, "Performance Lags"},
    {LogEntryCode::TransitiontoOK, "Transition to OK"},
    {LogEntryCode::TransitiontoNonCriticalfromOK, "Transition to Non-Critical from OK"},
    {LogEntryCode::TransitiontoCriticalfromlesssevere, "Transition to Critical from less severe"},
    {LogEntryCode::TransitiontoNonrecoverablefromlesssevere, "Transition to Non-recoverable from less severe"},
    {LogEntryCode::TransitiontoNonCriticalfrommoresevere, "Transition to Non-Critical from more severe"},
    {LogEntryCode::TransitiontoCriticalfromNonrecoverable, "Transition to Critical from Non-recoverable"},
    {LogEntryCode::TransitiontoNonrecoverable, "Transition to Non-recoverable"},
    {LogEntryCode::Monitor, "Monitor"},
    {LogEntryCode::Informational, "Informational"},
    {LogEntryCode::DeviceRemovedDeviceAbsent, "Device Removed / Device Absent"},
    {LogEntryCode::DeviceInsertedDevicePresent, "Device Inserted / Device Present"},
    {LogEntryCode::DeviceDisabled, "Device Disabled"},
    {LogEntryCode::DeviceEnabled, "Device Enabled"},
    {LogEntryCode::TransitiontoRunning, "Transition to Running"},
    {LogEntryCode::TransitiontoInTest, "Transition to In Test"},
    {LogEntryCode::TransitiontoPowerOff, "Transition to Power Off"},
    {LogEntryCode::TransitiontoOnLine, "Transition to On Line"},
    {LogEntryCode::TransitiontoOffLine, "Transition to Off Line"},
    {LogEntryCode::TransitiontoOffDuty, "Transition to Off Duty"},
    {LogEntryCode::TransitiontoDegraded, "Transition to Degraded"},
    {LogEntryCode::TransitiontoPowerSave, "Transition to Power Save"},
    {LogEntryCode::InstallError, "Install Error"},
    {LogEntryCode::FullyRedundant, "Fully Redundant"},
    {LogEntryCode::RedundancyLost, "Redundancy Lost"},
    {LogEntryCode::RedundancyDegraded, "Redundancy Degraded"},
    {LogEntryCode::NonredundantSufficientResourcesfromRedundant, "Non-redundant:Sufficient Resources from Redundant"},
    {LogEntryCode::NonredundantSufficientResourcesfromInsufficientResources, "Non-redundant:Sufficient Resources from Insufficient Resources"},
    {LogEntryCode::NonredundantInsufficientResources, "Non-redundant:Insufficient Resources"},
    {LogEntryCode::RedundancyDegradedfromFullyRedundant, "Redundancy Degraded from Fully Redundant"},
    {LogEntryCode::RedundancyDegradedfromNonredundant, "Redundancy Degraded from Non-redundant"},
    {LogEntryCode::D0PowerState, "D0 Power State"},
    {LogEntryCode::D1PowerState, "D1 Power State"},
    {LogEntryCode::D2PowerState, "D2 Power State"},
    {LogEntryCode::D3PowerState, "D3 Power State"},
    {LogEntryCode::OEM, "OEM"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(SensorType, {
    {SensorType::Invalid, "Invalid"},
    {SensorType::PlatformSecurityViolationAttempt, "Platform Security Violation Attempt"},
    {SensorType::Temperature, "Temperature"},
    {SensorType::Voltage, "Voltage"},
    {SensorType::Current, "Current"},
    {SensorType::Fan, "Fan"},
    {SensorType::PhysicalChassisSecurity, "Physical Chassis Security"},
    {SensorType::Processor, "Processor"},
    {SensorType::PowerSupplyConverter, "Power Supply / Converter"},
    {SensorType::PowerUnit, "PowerUnit"},
    {SensorType::CoolingDevice, "CoolingDevice"},
    {SensorType::OtherUnitsbasedSensor, "Other Units-based Sensor"},
    {SensorType::Memory, "Memory"},
    {SensorType::DriveSlotBay, "Drive Slot/Bay"},
    {SensorType::POSTMemoryResize, "POST Memory Resize"},
    {SensorType::SystemFirmwareProgress, "System Firmware Progress"},
    {SensorType::EventLoggingDisabled, "Event Logging Disabled"},
    {SensorType::SystemEvent, "System Event"},
    {SensorType::CriticalInterrupt, "Critical Interrupt"},
    {SensorType::ButtonSwitch, "Button/Switch"},
    {SensorType::ModuleBoard, "Module/Board"},
    {SensorType::MicrocontrollerCoprocessor, "Microcontroller/Coprocessor"},
    {SensorType::AddinCard, "Add-in Card"},
    {SensorType::Chassis, "Chassis"},
    {SensorType::ChipSet, "ChipSet"},
    {SensorType::OtherFRU, "Other FRU"},
    {SensorType::CableInterconnect, "Cable/Interconnect"},
    {SensorType::Terminator, "Terminator"},
    {SensorType::SystemBootRestart, "SystemBoot/Restart"},
    {SensorType::BootError, "Boot Error"},
    {SensorType::BaseOSBootInstallationStatus, "BaseOSBoot/InstallationStatus"},
    {SensorType::OSStopShutdown, "OS Stop/Shutdown"},
    {SensorType::SlotConnector, "Slot/Connector"},
    {SensorType::SystemACPIPowerState, "System ACPI PowerState"},
    {SensorType::Watchdog, "Watchdog"},
    {SensorType::PlatformAlert, "Platform Alert"},
    {SensorType::EntityPresence, "Entity Presence"},
    {SensorType::MonitorASICIC, "Monitor ASIC/IC"},
    {SensorType::LAN, "LAN"},
    {SensorType::ManagementSubsystemHealth, "Management Subsystem Health"},
    {SensorType::Battery, "Battery"},
    {SensorType::SessionAudit, "Session Audit"},
    {SensorType::VersionChange, "Version Change"},
    {SensorType::FRUState, "FRUState"},
    {SensorType::OEM, "OEM"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(LogDiagnosticDataTypes, {
    {LogDiagnosticDataTypes::Invalid, "Invalid"},
    {LogDiagnosticDataTypes::Manager, "Manager"},
    {LogDiagnosticDataTypes::PreOS, "PreOS"},
    {LogDiagnosticDataTypes::OS, "OS"},
    {LogDiagnosticDataTypes::OEM, "OEM"},
    {LogDiagnosticDataTypes::CPER, "CPER"},
    {LogDiagnosticDataTypes::CPERSection, "CPERSection"},
    {LogDiagnosticDataTypes::Device, "Device"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(OriginatorTypes, {
    {OriginatorTypes::Invalid, "Invalid"},
    {OriginatorTypes::Client, "Client"},
    {OriginatorTypes::Internal, "Internal"},
    {OriginatorTypes::SupportingService, "SupportingService"},
});

NLOHMANN_JSON_SERIALIZE_ENUM(CXLEntryType, {
    {CXLEntryType::Invalid, "Invalid"},
    {CXLEntryType::DynamicCapacity, "DynamicCapacity"},
    {CXLEntryType::Informational, "Informational"},
    {CXLEntryType::Warning, "Warning"},
    {CXLEntryType::Failure, "Failure"},
    {CXLEntryType::Fatal, "Fatal"},
});

}
// clang-format on
