// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: Copyright OpenBMC Authors
#pragma once
#include "http_response.hpp"
#include "http_utility.hpp"
#include "sessions.hpp"

#include <boost/beast/http/field.hpp>
#include <boost/beast/http/status.hpp>
#include <boost/url/format.hpp>
#include <boost/url/url.hpp>

#include <string_view>

namespace forward_unauthorized
{

// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static bool hasWebuiRoute = false;

inline void sendUnauthorized(std::string_view url,
                             std::string_view xRequestedWith,
                             std::string_view accept, crow::Response& res)
{
    // If it's a browser connecting, don't send the HTTP authenticate
    // header, to avoid possible CSRF attacks with basic auth
    if (http_helpers::isContentTypeAllowed(
            accept, http_helpers::ContentType::HTML, false /*allowWildcard*/))
    {
        // If we have a webui installed, redirect to that login page
        if (hasWebuiRoute)
        {
            boost::urls::url forward =
                boost::urls::format("/?next={}#/login", url);
            res.result(boost::beast::http::status::temporary_redirect);
            res.addHeader(boost::beast::http::field::location,
                          forward.buffer());
            return;
        }
        // If we don't have a webui installed, just return an unauthorized
        // body
        res.result(boost::beast::http::status::unauthorized);
        res.write(
            "No authentication provided, and no login UI present to forward to.");
        return;
    }

    res.result(boost::beast::http::status::unauthorized);

    // XHR requests from a browser will set the X-Requested-With header when
    // doing their requests, even though they might not be requesting html.
    if (!xRequestedWith.empty())
    {
        return;
    }
    // if basic auth is disabled, don't propose it.
    if (!persistent_data::SessionStore::getInstance()
             .getAuthMethodsConfig()
             .basic)
    {
        return;
    }
    res.addHeader(boost::beast::http::field::www_authenticate, "Basic");
}
} // namespace forward_unauthorized
