# Internationalization

The OpenBMC Web UI implements internationalization and separates the language-
specific parts of the interface from the rest of the code, so they can be easily
replaced. The OpenBMC Web UI uses the following library for
internationalization:

- [Vue I18n](https://kazupon.github.io/vue-i18n/introduction.html)

## Key naming convention

The OpenBMC Web UI follows the following key naming conventions:

- Page specific labels should be nested in an object with a key prefixed `page`
  followed by the page title. Formatting in this manner provides a systematic
  structure and improves readability of the language file.
  - e.g. `pageLocalUserManagement.editUser`
- Any 'major' child components should be nested inside page specific objects
  (ex. table, modal)
  - e.g. `pageEventLogs.table.eventType`
- Avoid any complex linked locale messages.
- Alphabetize object keys. This helps in locating the keys.
- We use the `$t()` function in markup and `this.$t` in scripts (which Vue I18n
  provides to our components) for outputting translation messages.

## Using the Vue I18n plugin

- A new `src/i18n.js` file is added and it registers Vue I18n as a plugin to our
  Vue instance via the `Vue.use()` function.
- The CLI creates a `src/locales/en-US.json` file, which contains our default
  translation messages.
- The keys are placed in the `src/locales/en-US.json` file and then the `$t()`
  function is used to output the translation messages.
- After adding or removing calls to `$t` please run this to ensure consistent
  English translation (note: using variable expansion in key names is not
  handled automatically, you need to manually check them):

```bash
node node_modules/vue-i18n-extract/bin/vue-i18n-extract.js -v 'src/**/*.?(js|vue)' -l 'src/locales/en-US.json'
```

- If you're working on updating a translation for another language (e.g.
  Russian), run this to see the omissions (as well as cruft) and add the
  necessary keys automatically:

```bash
node node_modules/vue-i18n-extract/bin/vue-i18n-extract.js -v 'src/**/*.?(js|vue)' -l 'src/locales/ru-RU.json' -a
```

```json
"pageDumps": {
  "dumpsAvailableOnBmc": "Dumps available on BMC"
}
```

```Vue
<page-section :section-title="$t('pageDumps.dumpsAvailableOnBmc')">
```

- Vue I18n’s `$tc()` function can help with displaying plurals.
  [Learn more about pluralization.](https://kazupon.github.io/vue-i18n/guide/pluralization.html)

```json
"modal": {
  "deleteDump": "Delete dump | Delete dumps"
}
```

```JS
this.$bvModal
  .msgBoxConfirm(this.$tc('pageDumps.modal.deleteDumpConfirmation'), {
   title: this.$tc('pageDumps.modal.deleteDump'),
   okTitle: this.$tc('pageDumps.modal.deleteDump'),
   cancelTitle: this.$t('global.action.cancel'),
   autoFocusButton: 'ok',
  })
```

## Vendor overlays (environment-specific translations)

To keep the base translation files vendor-neutral, vendor-specific strings live
under `src/env/locales/<envName>/`.

- Place shared vendor strings in the vendor root folder (e.g.,
  `src/env/locales/vendor/`)
- Place project-specific overrides in the variant folder when needed (e.g.,
  `src/env/locales/vendor-variant/`)
- Merge order at runtime:
  1. Base locales from `src/locales/` (auto-discovered)
  2. Vendor root overlays (e.g., `src/env/locales/vendor/`)
  3. Variant overlays (e.g., `src/env/locales/vendor-variant/`)
     - Variant keys overwrite vendor root keys on conflict

Notes:

- All JSON files under `src/locales/` are bundled automatically.
- All JSON files under `src/env/locales/` that match the active environment are
  also bundled.
- If multiple vendor projects share strings, prefer the vendor root folder so
  variants don’t duplicate content.

Example: moving vendor-only dump type labels

```json
// src/locales/en-US.json (base)
{
  "pageDumps": {
    "dumpTypes": {}
  }
}

// src/env/locales/nvidia/en-US.json (overlay)
{
  "pageDumps": {
    "dumpTypes": {
      "hmcDump": "HMC dump",
      "bmcDump": "BMC dump",
      "systemBmcDump": "System [BMC] dump (disruptive)",
      "systemHgxDump": "System [HGX] dump (disruptive)"
    }
  }
}
```

### Locale codes

We support aliasing short codes to our canonical locales:

- `en` → `en-US`
- `ru` → `ru-RU`
- `zh` → `zh-CN`
- `ka` → `ka-GE`

If a short code is stored (e.g., in localStorage), it will be normalized at app
startup.
